const fs = require('fs');

// Fonction pour corriger le fichier geoloc.js
function fixGeolocFile() {
  const filePath = '/home/project/js/geoloc.js';
  
  if (!fs.existsSync(filePath)) {
    console.error(`Fichier non trouvé: ${filePath}`);
    return;
  }
  
  try {
    // Lire le contenu original
    const originalContent = fs.readFileSync(filePath, 'utf8');
    
    // Créer une version corrigée complète du fichier
    const correctedContent = `document.addEventListener('DOMContentLoaded', () => {
    console.log("JS chargé (geoloc.js)");

    const geolocateButton = document.querySelector('.geolocation-button');
    const farmerList = document.querySelector('.farmer-list');
    const distancesInfo = document.querySelector('.distances-info');

    if (!geolocateButton || !farmerList) {
        console.error("Éléments DOM manquants");
        return;
    }

    // Fonction pour calculer la distance (fallback)
    function calculateHaversineDistance(lat1, lon1, lat2, lon2) {
        const R = 6371; // Rayon de la Terre en km
        const dLat = (lat2 - lat1) * Math.PI / 180;
        const dLon = (lon2 - lon1) * Math.PI / 180;
        const a = Math.sin(dLat/2) * Math.sin(dLat/2) +
                Math.cos(lat1 * Math.PI / 180) * Math.cos(lat2 * Math.PI / 180) * 
                Math.sin(dLon/2) * Math.sin(dLon/2);
        const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1-a));
        return R * c;
    }

    geolocateButton.addEventListener('click', () => {
        if (!navigator.geolocation) {
            alert("Géolocalisation non supportée par votre navigateur.");
            return;
        }

        geolocateButton.disabled = true;
        geolocateButton.innerHTML = '⏳ Localisation...';

        navigator.geolocation.getCurrentPosition(async (position) => {
            const userLat = position.coords.latitude;
            const userLng = position.coords.longitude;
            console.log("Position utilisateur:", userLat, userLng);

            if (distancesInfo) {
                distancesInfo.style.display = 'block';
            }

            const items = document.querySelectorAll('.farmer-item');
            const distances = [];
            let successCount = 0;
            let fallbackCount = 0;

            // Traiter chaque agriculteur
            for (const item of items) {
                const id = item.getAttribute('data-id');
                const coords = item.getAttribute('data-coordinates');
                
                if (!coords) {
                    console.warn(\`Agriculteur ID \${id}: Pas de coordonnées disponibles\`);
                    continue;
                }
                
                const [farmLat, farmLng] = coords.split(',').map(parseFloat);
                
                // Vérifier que les coordonnées sont valides
                if (isNaN(farmLat) || isNaN(farmLng)) {
                    console.warn(\`Coordonnées invalides pour l'agriculteur ID \${id}: \${coords}\`);
                    continue;
                }
                
                console.log(\`Calcul de distance pour agriculteur \${id}: [\${farmLat}, \${farmLng}]\`);
                
                // Calculer d'abord la distance comme fallback
                const haversineDistance = calculateHaversineDistance(userLat, userLng, farmLat, farmLng);
                console.log(\`Distance "directe" calculée pour ID \${id}: \${haversineDistance.toFixed(1)} km\`);
                
                let distanceKm = haversineDistance; // Utiliser la distance "directe" par défaut
                let usesFallback = true; // Par défaut, on utilise le fallback

                try {
                    // Essayer d'obtenir la distance routière
                    // Inverser lat/lng pour OpenRouteService qui attend [longitude, latitude]
                    const url = \`/modules/mod_agriculteur_liste/api/get-distance.php?start=\${userLng},\${userLat}&end=\${farmLng},\${farmLat}\`;
                    console.log(\`Appel API pour ID \${id}:\`, url);
                    
                    const response = await fetch(url);
                    const responseText = await response.text();
                    
                    console.log(\`Réponse brute pour ID \${id}:\`, responseText);
                    
                    // Essayer de parser la réponse JSON
                    let data;
                    try {
                        data = JSON.parse(responseText);
                    } catch (parseError) {
                        console.error(\`Erreur de parsing JSON pour ID \${id}:\`, parseError);
                        throw new Error(\`Réponse non-JSON: \${responseText.substring(0, 100)}...\`);
                    }
                    
                    console.log(\`Réponse API pour ID \${id}:\`, data);
                    
                    // Vérifier si la réponse contient une erreur
                    if (data.error) {
                        throw new Error(\`Erreur API: \${data.error}\`);
                    }
                    
                    // Extraire la distance de la réponse
                    if (data.features && 
                        data.features[0] && 
                        data.features[0].properties && 
                        data.features[0].properties.summary && 
                        typeof data.features[0].properties.summary.distance === 'number') {
                        
                        // Convertir de mètres en kilomètres
                        distanceKm = data.features[0].properties.summary.distance / 1000;
                        console.log(\`Distance routière calculée pour ID \${id}: \${distanceKm.toFixed(1)} km\`);
                        usesFallback = false;
                        successCount++;
                    } else {
                        console.warn(\`Structure de réponse inattendue pour ID \${id}, utilisation du fallback\`);
                        fallbackCount++;
                    }
                } catch (error) {
                    console.error(\`Erreur lors du calcul de la distance pour ID \${id}:\`, error);
                    // On garde la distance calculée précédemment
                    console.log(\`Utilisation du fallback pour ID \${id}: \${distanceKm.toFixed(1)} km\`);
                    fallbackCount++;
                }

                // Mettre à jour l'affichage de la distance
                const distEl = document.getElementById('distance-' + id);
                if (distEl) {
                    const valueEl = distEl.querySelector('.distance-value');
                    if (valueEl) {
                        // CORRECTION: Forcer la mise à jour du contenu textuel
                        valueEl.textContent = distanceKm.toFixed(1);
                        
                        // Forcer un rafraîchissement du DOM
                        setTimeout(() => {
                            valueEl.textContent = distanceKm.toFixed(1);
                        }, 10);
                        
                        // Ajouter une classe pour indiquer si c'est une distance routière ou "directe"
                        if (usesFallback) {
                            distEl.classList.add('fallback-distance');
                        } else {
                            distEl.classList.add('road-distance');
                        }
                        distEl.style.display = 'block';
                        console.log(\`Mise à jour de l'affichage pour ID \${id}: \${distanceKm.toFixed(1)} km\`);
                    }
                }

                // Mettre à jour aussi dans la vue grille si elle existe
                const gridDistEl = document.getElementById('grid-distance-' + id);
                if (gridDistEl) {
                    const gridValueEl = gridDistEl.querySelector('.grid-distance-value');
                    if (gridValueEl) {
                        // CORRECTION: Forcer la mise à jour du contenu textuel
                        gridValueEl.textContent = distanceKm.toFixed(1);
                        
                        // Forcer un rafraîchissement du DOM
                        setTimeout(() => {
                            gridValueEl.textContent = distanceKm.toFixed(1);
                        }, 10);
                        
                        // Ajouter une classe pour indiquer si c'est une distance routière ou "directe"
                        if (usesFallback) {
                            gridDistEl.classList.add('fallback-distance');
                        } else {
                            gridDistEl.classList.add('road-distance');
                        }
                    }
                }

                // Stocker pour le tri
                distances.push({ id, distance: distanceKm, element: item });
                item.setAttribute('data-distance', distanceKm);
            }

            console.log(\`Statistiques: \${successCount} distances routières calculées, \${fallbackCount} distances directes utilisées\`);

            // Trier les éléments par distance
            distances.sort((a, b) => a.distance - b.distance);
            
            // Réorganiser les éléments dans le DOM
            distances.forEach(d => {
                const parent = d.element.parentNode;
                if (parent) {
                    farmerList.appendChild(parent);
                }
            });

            // Réinitialiser le bouton
            geolocateButton.disabled = false;
            geolocateButton.innerHTML = '📍 Trouver les agriculteurs proches de moi';
            
            console.log("Calcul des distances terminé");
        }, (error) => {
            console.error("Erreur de géolocalisation:", error);
            alert("Impossible d'obtenir votre position.");
            geolocateButton.disabled = false;
            geolocateButton.innerHTML = '📍 Trouver les agriculteurs proches de moi';
        }, {
            enableHighAccuracy: true,
            timeout: 10000,
            maximumAge: 0
        });
    });
});`;
    
    // Écrire le contenu corrigé
    fs.writeFileSync(filePath, correctedContent, 'utf8');
    console.log(`✅ Correction complète appliquée à: ${filePath}`);
    
  } catch (err) {
    console.error(`Erreur lors de la correction de ${filePath}: ${err}`);
  }
}

// Fonction pour corriger le fichier main.js
function fixMainFile() {
  const filePath = '/home/project/tmpl/assets/main.js';
  
  if (!fs.existsSync(filePath)) {
    console.error(`Fichier non trouvé: ${filePath}`);
    return;
  }
  
  try {
    // Lire le contenu original
    let content = fs.readFileSync(filePath, 'utf8');
    
    // Rechercher et corriger les occurrences problématiques
    
    // 1. Corriger "// Fonction pour calculer la "distance directe" (formule Haversine)"
    content = content.replace(
      /\/\/ Fonction pour calculer la "distance directe" \(formule Haversine\)/g,
      '// Fonction pour calculer la distance "directe" (formule Haversine)'
    );
    
    // 2. Corriger "// Fallback: "distance directe""
    content = content.replace(
      /\/\/ Fallback: "distance directe"/g,
      '// Fallback: distance directe'
    );
    
    // 3. Corriger les commentaires contenant "distance directe"
    content = content.replace(
      /"distance directe"/g,
      'distance directe'
    );
    
    // Écrire le contenu corrigé
    fs.writeFileSync(filePath, content, 'utf8');
    console.log(`✅ Corrections appliquées à: ${filePath}`);
    
  } catch (err) {
    console.error(`Erreur lors de la correction de ${filePath}: ${err}`);
  }
}

// Exécuter les corrections
console.log("Application des corrections finales...");
fixGeolocFile();
fixMainFile();
console.log("Corrections finales terminées.");
